'use strict';

/* --------------------------------------------------------------
 datatable_default_actions.js 2016-10-12
 Gambio GmbH
 http://www.gambio.de
 Copyright (c) 2016 Gambio GmbH
 Released under the GNU General Public License (Version 2)
 [http://www.gnu.org/licenses/gpl-2.0.html]
 --------------------------------------------------------------
 */

/**
 * ## Enable Default Dropdown Actions
 *
 * This extension will handle the "defaultRowAction" and "defaultBulkAction" data attributes of the table upon
 * initialization or user click.
 *
 * ### Options
 *
 * **Default Row Action | `data-datatable_default_actions-row` | String | Required**
 *
 * Provide the default row action. This will automatically be mapped to the defaultRowAction data value of the table.
 *
 * **Default Bulk Action | `data-datatable_default_actions-bulk` | String | Required**
 *
 * Provide the default bulk action. This will automatically be mapped to the defaultBulkAction data value of the table.
 *
 * **Bulk Action Selector | `data-datatable_default_actions-bulk-action-selector` | String | Optional**
 *
 * Provide a selector for the bulk action dropdown widget. The default value is '.bulk-action'.
 *
 * ### Methods
 *
 * **Ensure Default Task**
 *
 * This method will make sure that there is a default task selected. Call it after you setup the row or bulk dropdown
 * actions. Sometimes the user_configuration db value might contain a default value that is not present in the dropdowns
 * anymore (e.g. removed module). In order to make sure that there will always be a default value use this method after
 * creating the dropdown actions and it will use the first dropdown action as default if needed.
 *
 * ```javascript
 * // Ensure default row actions.
 * $('.table-main').datatable_default_actions('ensure', 'row');
 *
 * // Ensure default bulk actions.
 * $('.table-main').datatable_default_actions('ensure', 'bulk');
 * ```
 *
 * @module Admin/extensions/datatable_default_actions
 */
gx.extensions.module('datatable_default_actions', [gx.source + '/libs/button_dropdown'], function (data) {

    'use strict';

    // ------------------------------------------------------------------------
    // VARIABLES
    // ------------------------------------------------------------------------


    /**
     * Module Selector
     *
     * @type {jQuery}
     */

    var $this = $(this);

    /**
     * Default Options
     *
     * @type {Object}
     */
    var defaults = {
        bulkActionSelector: '.bulk-action'
    };

    /**
     * Final Options
     *
     * @type {Object}
     */
    var options = $.extend(true, {}, defaults, data);

    /**
     * Module Instance
     *
     * @type {Object}
     */
    var module = {};

    // ------------------------------------------------------------------------
    // FUNCTIONS
    // ------------------------------------------------------------------------

    /**
     * Ensure that there will be a default action in the row or bulk dropdowns.
     *
     * @param {String} type Can be whether 'row' or 'bulk'.
     */
    function _ensure(type) {
        var $table = $(this);

        switch (type) {
            case 'row':
                var $rowActions = $table.find('tbody .btn-group.dropdown');
                // debugger;
                $rowActions.each(function () {
                    if ($(this).find('button:empty').length) {
                        var $actionLink = $(this).find('ul li:first a');
                        jse.libs.button_dropdown.setDefaultAction($(this), $actionLink);
                    }
                });

                break;

            case 'bulk':
                var $bulkAction = $(options.bulkActionSelector);

                if ($bulkAction.find('button:first').text() === '') {
                    var $actionLink = $bulkAction.find('ul li:first a');
                    jse.libs.button_dropdown.setDefaultAction($bulkAction, $actionLink);
                }

                break;

            default:
                throw new Error('Invalid "ensure" type given (expected "row" or "bulk" got : "' + type + '").');
        }
    }

    /**
     * On Button Drodpown Action Click
     *
     * Update the defaultBulkAction and defaultRowAction data attributes.
     */
    function _onButtonDropdownActionClick() {
        var property = $(this).parents('.btn-group')[0] === $(options.bulkActionSelector)[0] ? 'defaultBulkAction' : 'defaultRowAction';

        $this.data(property, $(this).data('configurationValue'));
    }

    // ------------------------------------------------------------------------
    // INITIALIZATION
    // ------------------------------------------------------------------------

    module.init = function (done) {
        $this.data({
            defaultRowAction: options.row,
            defaultBulkAction: options.bulk
        });

        $this.on('click', '.btn-group.dropdown a', _onButtonDropdownActionClick);
        $('body').on('click', options.bulkActionSelector, _onButtonDropdownActionClick);

        // Bind module api to jQuery object.
        $.fn.extend({
            datatable_default_actions: function datatable_default_actions(action) {
                switch (action) {
                    case 'ensure':
                        for (var _len = arguments.length, args = Array(_len > 1 ? _len - 1 : 0), _key = 1; _key < _len; _key++) {
                            args[_key - 1] = arguments[_key];
                        }

                        return _ensure.apply(this, args);
                }
            }
        });

        done();
    };

    return module;
});
//# sourceMappingURL=data:application/json;charset=utf8;base64,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
